/*
 * 3GPP AKA - Milenage algorithm (3GPP TS 35.205, .206, .207, .208)
 * Copyright (c) 2006-2007 <j@w1.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Alternatively, this software may be distributed under the terms of BSD
 * license.
 *
 * See README and COPYING for more details.
 *
 * This file implements an example authentication algorithm defined for 3GPP
 * AKA. This can be used to implement a simple HLR/AuC into hlr_auc_gw to allow
 * EAP-AKA to be tested properly with real USIM cards.
 *
 * This implementations assumes that the r1..r5 and c1..c5 constants defined in
 * TS 35.206 are used, i.e., r1=64, r2=0, r3=32, r4=64, r5=96, c1=00..00,
 * c2=00..01, c3=00..02, c4=00..04, c5=00..08. The block cipher is assumed to
 * be AES (Rijndael).
 */

#include "includes.h"

#include "common.h"
#include "milenage.h"
#include "aes_wrap.h"


/**
 * milenage_f1 - Milenage f1 and f1* algorithms
 * @opc: OPc = 128-bit value derived from OP and K
 * @k: K = 128-bit subscriber key
 * @_rand: RAND = 128-bit random challenge
 * @sqn: SQN = 48-bit sequence number
 * @amf: AMF = 16-bit authentication management field
 * @mac_a: Buffer for MAC-A = 64-bit network authentication code, or %NULL
 * @mac_s: Buffer for MAC-S = 64-bit resync authentication code, or %NULL
 * Returns: 0 on success, -1 on failure
 */
static int milenage_f1(const u8 *opc, const u8 *k, const u8 *_rand,
		       const u8 *sqn, const u8 *amf, u8 *mac_a, u8 *mac_s)
{
	u8 tmp1[16], tmp2[16], tmp3[16];
	int i;

	/* tmp1 = TEMP = E_K(RAND XOR OP_C) */
	for (i = 0; i < 16; i++)
		tmp1[i] = _rand[i] ^ opc[i];
	if (aes_128_encrypt_block(k, tmp1, tmp1))
		return -1;

	/* tmp2 = IN1 = SQN || AMF || SQN || AMF */
	os_memcpy(tmp2, sqn, 6);
	os_memcpy(tmp2 + 6, amf, 2);
	os_memcpy(tmp2 + 8, tmp2, 8);

	/* OUT1 = E_K(TEMP XOR rot(IN1 XOR OP_C, r1) XOR c1) XOR OP_C */

	/* rotate (tmp2 XOR OP_C) by r1 (= 0x40 = 8 bytes) */
	for (i = 0; i < 16; i++)
		tmp3[(i + 8) % 16] = tmp2[i] ^ opc[i];
	/* XOR with TEMP = E_K(RAND XOR OP_C) */
	for (i = 0; i < 16; i++)
		tmp3[i] ^= tmp1[i];
	/* XOR with c1 (= ..00, i.e., NOP) */

	/* f1 || f1* = E_K(tmp3) XOR OP_c */
	if (aes_128_encrypt_block(k, tmp3, tmp1))
		return -1;
	for (i = 0; i < 16; i++)
		tmp1[i] ^= opc[i];
	if (mac_a)
		os_memcpy(mac_a, tmp1, 8); /* f1 */
	if (mac_s)
		os_memcpy(mac_s, tmp1 + 8, 8); /* f1* */
	return 0;
}


/**
 * milenage_f2345 - Milenage f2, f3, f4, f5, f5* algorithms
 * @opc: OPc = 128-bit value derived from OP and K
 * @k: K = 128-bit subscriber key
 * @_rand: RAND = 128-bit random challenge
 * @res: Buffer for RES = 64-bit signed response (f2), or %NULL
 * @ck: Buffer for CK = 128-bit confidentiality key (f3), or %NULL
 * @ik: Buffer for IK = 128-bit integrity key (f4), or %NULL
 * @ak: Buffer for AK = 48-bit anonymity key (f5), or %NULL
 * @akstar: Buffer for AK = 48-bit anonymity key (f5*), or %NULL
 * Returns: 0 on success, -1 on failure
 */
static int milenage_f2345(const u8 *opc, const u8 *k, const u8 *_rand,
			  u8 *res, u8 *ck, u8 *ik, u8 *ak, u8 *akstar)
{
	u8 tmp1[16], tmp2[16], tmp3[16];
	int i;

	/* tmp2 = TEMP = E_K(RAND XOR OP_C) */
	for (i = 0; i < 16; i++)
		tmp1[i] = _rand[i] ^ opc[i];
	if (aes_128_encrypt_block(k, tmp1, tmp2))
		return -1;

	/* OUT2 = E_K(rot(TEMP XOR OP_C, r2) XOR c2) XOR OP_C */
	/* OUT3 = E_K(rot(TEMP XOR OP_C, r3) XOR c3) XOR OP_C */
	/* OUT4 = E_K(rot(TEMP XOR OP_C, r4) XOR c4) XOR OP_C */
	/* OUT5 = E_K(rot(TEMP XOR OP_C, r5) XOR c5) XOR OP_C */

	/* f2 and f5 */
	/* rotate by r2 (= 0, i.e., NOP) */
	for (i = 0; i < 16; i++)
		tmp1[i] = tmp2[i] ^ opc[i];
	tmp1[15] ^= 1; /* XOR c2 (= ..01) */
	/* f5 || f2 = E_K(tmp1) XOR OP_c */
	if (aes_128_encrypt_block(k, tmp1, tmp3))
		return -1;
	for (i = 0; i < 16; i++)
		tmp3[i] ^= opc[i];
	if (res)
		os_memcpy(res, tmp3 + 8, 8); /* f2 */
	if (ak)
		os_memcpy(ak, tmp3, 6); /* f5 */

	/* f3 */
	if (ck) {
		/* rotate by r3 = 0x20 = 4 bytes */
		for (i = 0; i < 16; i++)
			tmp1[(i + 12) % 16] = tmp2[i] ^ opc[i];
		tmp1[15] ^= 2; /* XOR c3 (= ..02) */
		if (aes_128_encrypt_block(k, tmp1, ck))
			return -1;
		for (i = 0; i < 16; i++)
			ck[i] ^= opc[i];
	}

	/* f4 */
	if (ik) {
		/* rotate by r4 = 0x40 = 8 bytes */
		for (i = 0; i < 16; i++)
			tmp1[(i + 8) % 16] = tmp2[i] ^ opc[i];
		tmp1[15] ^= 4; /* XOR c4 (= ..04) */
		if (aes_128_encrypt_block(k, tmp1, ik))
			return -1;
		for (i = 0; i < 16; i++)
			ik[i] ^= opc[i];
	}

	/* f5* */
	if (akstar) {
		/* rotate by r5 = 0x60 = 12 bytes */
		for (i = 0; i < 16; i++)
			tmp1[(i + 4) % 16] = tmp2[i] ^ opc[i];
		tmp1[15] ^= 8; /* XOR c5 (= ..08) */
		if (aes_128_encrypt_block(k, tmp1, tmp1))
			return -1;
		for (i = 0; i < 6; i++)
			akstar[i] = tmp1[i] ^ opc[i];
	}

	return 0;
}


/**
 * milenage_generate - Generate AKA AUTN,IK,CK,RES
 * @opc: OPc = 128-bit operator variant algorithm configuration field (encr.)
 * @amf: AMF = 16-bit authentication management field
 * @k: K = 128-bit subscriber key
 * @sqn: SQN = 48-bit sequence number
 * @_rand: RAND = 128-bit random challenge
 * @autn: Buffer for AUTN = 128-bit authentication token
 * @ik: Buffer for IK = 128-bit integrity key (f4), or %NULL
 * @ck: Buffer for CK = 128-bit confidentiality key (f3), or %NULL
 * @res: Buffer for RES = 64-bit signed response (f2), or %NULL
 * @res_len: Max length for res; set to used length or 0 on failure
 */
void milenage_generate(const u8 *opc, const u8 *amf, const u8 *k,
		       const u8 *sqn, const u8 *_rand, u8 *autn, u8 *ik,
		       u8 *ck, u8 *res, size_t *res_len)
{
	int i;
	u8 mac_a[8], ak[6];

	if (*res_len < 8) {
		*res_len = 0;
		return;
	}
	if (milenage_f1(opc, k, _rand, sqn, amf, mac_a, NULL) ||
	    milenage_f2345(opc, k, _rand, res, ck, ik, ak, NULL)) {
		*res_len = 0;
		return;
	}
	*res_len = 8;

	/* AUTN = (SQN ^ AK) || AMF || MAC */
	for (i = 0; i < 6; i++)
		autn[i] = sqn[i] ^ ak[i];
	os_memcpy(autn + 6, amf, 2);
	os_memcpy(autn + 8, mac_a, 8);
}


/**
 * milenage_auts - Milenage AUTS validation
 * @opc: OPc = 128-bit operator variant algorithm configuration field (encr.)
 * @k: K = 128-bit subscriber key
 * @_rand: RAND = 128-bit random challenge
 * @auts: AUTS = 112-bit authentication token from client
 * @sqn: Buffer for SQN = 48-bit sequence number
 * Returns: 0 = success (sqn filled), -1 on failure
 */
int milenage_auts(const u8 *opc, const u8 *k, const u8 *_rand, const u8 *auts,
		  u8 *sqn)
{
	u8 amf[2] = { 0x00, 0x00 }; /* TS 33.102 v7.0.0, 6.3.3 */
	u8 ak[6], mac_s[8];
	int i;

	if (milenage_f2345(opc, k, _rand, NULL, NULL, NULL, NULL, ak))
		return -1;
	for (i = 0; i < 6; i++)
		sqn[i] = auts[i] ^ ak[i];
	if (milenage_f1(opc, k, _rand, sqn, amf, NULL, mac_s) ||
	    memcmp(mac_s, auts + 6, 8) != 0)
		return -1;
	return 0;
}


/**
 * gsm_milenage - Generate GSM-Milenage (3GPP TS 55.205) authentication triplet
 * @opc: OPc = 128-bit operator variant algorithm configuration field (encr.)
 * @k: K = 128-bit subscriber key
 * @_rand: RAND = 128-bit random challenge
 * @sres: Buffer for SRES = 32-bit SRES
 * @kc: Buffer for Kc = 64-bit Kc
 * Returns: 0 on success, -1 on failure
 */
int gsm_milenage(const u8 *opc, const u8 *k, const u8 *_rand, u8 *sres, u8 *kc)
{
	u8 res[8], ck[16], ik[16];
	int i;

	if (milenage_f2345(opc, k, _rand, res, ck, ik, NULL, NULL))
		return -1;

	for (i = 0; i < 8; i++)
		kc[i] = ck[i] ^ ck[i + 8] ^ ik[i] ^ ik[i + 8];

#ifdef GSM_MILENAGE_ALT_SRES
	os_memcpy(sres, res, 4);
#else /* GSM_MILENAGE_ALT_SRES */
	for (i = 0; i < 4; i++)
		sres[i] = res[i] ^ res[i + 4];
#endif /* GSM_MILENAGE_ALT_SRES */
	return 0;
}


/**
 * milenage_generate - Generate AKA AUTN,IK,CK,RES
 * @opc: OPc = 128-bit operator variant algorithm configuration field (encr.)
 * @k: K = 128-bit subscriber key
 * @sqn: SQN = 48-bit sequence number
 * @_rand: RAND = 128-bit random challenge
 * @autn: AUTN = 128-bit authentication token
 * @ik: Buffer for IK = 128-bit integrity key (f4), or %NULL
 * @ck: Buffer for CK = 128-bit confidentiality key (f3), or %NULL
 * @res: Buffer for RES = 64-bit signed response (f2), or %NULL
 * @res_len: Variable that will be set to RES length
 * @auts: 112-bit buffer for AUTS
 * Returns: 0 on success, -1 on failure, or -2 on synchronization failure
 */
int milenage_check(const u8 *opc, const u8 *k, const u8 *sqn, const u8 *_rand,
		   const u8 *autn, u8 *ik, u8 *ck, u8 *res, size_t *res_len,
		   u8 *auts)
{
	int i;
	u8 mac_a[8], ak[6], rx_sqn[6];
	const u8 *amf;

	wpa_hexdump(MSG_DEBUG, "Milenage: AUTN", autn, 16);
	wpa_hexdump(MSG_DEBUG, "Milenage: RAND", _rand, 16);

	if (milenage_f2345(opc, k, _rand, res, ck, ik, ak, NULL))
		return -1;

	*res_len = 8;
	wpa_hexdump_key(MSG_DEBUG, "Milenage: RES", res, *res_len);
	wpa_hexdump_key(MSG_DEBUG, "Milenage: CK", ck, 16);
	wpa_hexdump_key(MSG_DEBUG, "Milenage: IK", ik, 16);
	wpa_hexdump_key(MSG_DEBUG, "Milenage: AK", ak, 6);

	/* AUTN = (SQN ^ AK) || AMF || MAC */
	for (i = 0; i < 6; i++)
		rx_sqn[i] = autn[i] ^ ak[i];
	wpa_hexdump(MSG_DEBUG, "Milenage: SQN", rx_sqn, 6);

	if (os_memcmp(rx_sqn, sqn, 6) <= 0) {
		u8 auts_amf[2] = { 0x00, 0x00 }; /* TS 33.102 v7.0.0, 6.3.3 */
		if (milenage_f2345(opc, k, _rand, NULL, NULL, NULL, NULL, ak))
			return -1;
		wpa_hexdump_key(MSG_DEBUG, "Milenage: AK*", ak, 6);
		for (i = 0; i < 6; i++)
			auts[i] = sqn[i] ^ ak[i];
		if (milenage_f1(opc, k, _rand, sqn, auts_amf, NULL, auts + 6))
			return -1;
		wpa_hexdump(MSG_DEBUG, "Milenage: AUTS", auts, 14);
		return -2;
	}

	amf = autn + 6;
	wpa_hexdump(MSG_DEBUG, "Milenage: AMF", amf, 2);
	if (milenage_f1(opc, k, _rand, rx_sqn, amf, mac_a, NULL))
		return -1;

	wpa_hexdump(MSG_DEBUG, "Milenage: MAC_A", mac_a, 8);

	if (os_memcmp(mac_a, autn + 8, 8) != 0) {
		wpa_printf(MSG_DEBUG, "Milenage: MAC mismatch");
		wpa_hexdump(MSG_DEBUG, "Milenage: Received MAC_A",
			    autn + 8, 8);
		return -1;
	}

	return 0;
}


#ifdef TEST_MAIN_MILENAGE

extern int wpa_debug_level;


/**
 * milenage_opc - Determine OPc from OP and K
 * @op: OP = 128-bit operator variant algorithm configuration field
 * @k: K = 128-bit subscriber key
 * @opc: Buffer for OPc = 128-bit value derived from OP and K
 */
static void milenage_opc(const u8 *op, const u8 *k, u8 *opc)
{
	int i;
	/* OP_C = OP XOR E_K(OP) */
	aes_128_encrypt_block(k, op, opc);
	for (i = 0; i < 16; i++)
		opc[i] ^= op[i];
}


struct gsm_milenage_test_set {
	u8 ki[16];
	u8 rand[16];
	u8 opc[16];
	u8 sres1[4];
	u8 sres2[4];
	u8 kc[8];
};

static const struct gsm_milenage_test_set gsm_test_sets[] =
{
	{
		/* 3GPP TS 55.205 v6.0.0 - Test Set 1 */
		{ 0x46, 0x5b, 0x5c, 0xe8, 0xb1, 0x99, 0xb4, 0x9f,
		  0xaa, 0x5f, 0x0a, 0x2e, 0xe2, 0x38, 0xa6, 0xbc },
		{ 0x23, 0x55, 0x3c, 0xbe, 0x96, 0x37, 0xa8, 0x9d,
		  0x21, 0x8a, 0xe6, 0x4d, 0xae, 0x47, 0xbf, 0x35 },
		{ 0xcd, 0x63, 0xcb, 0x71, 0x95, 0x4a, 0x9f, 0x4e,
		  0x48, 0xa5, 0x99, 0x4e, 0x37, 0xa0, 0x2b, 0xaf },
		{ 0x46, 0xf8, 0x41, 0x6a },
		{ 0xa5, 0x42, 0x11, 0xd5 },
		{ 0xea, 0xe4, 0xbe, 0x82, 0x3a, 0xf9, 0xa0, 0x8b }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 2 */
		{ 0xfe, 0xc8, 0x6b, 0xa6, 0xeb, 0x70, 0x7e, 0xd0,
		  0x89, 0x05, 0x75, 0x7b, 0x1b, 0xb4, 0x4b, 0x8f },
		{ 0x9f, 0x7c, 0x8d, 0x02, 0x1a, 0xcc, 0xf4, 0xdb,
		  0x21, 0x3c, 0xcf, 0xf0, 0xc7, 0xf7, 0x1a, 0x6a },
		{ 0x10, 0x06, 0x02, 0x0f, 0x0a, 0x47, 0x8b, 0xf6,
		  0xb6, 0x99, 0xf1, 0x5c, 0x06, 0x2e, 0x42, 0xb3 },
		{ 0x8c, 0x30, 0x8a, 0x5e },
		{ 0x80, 0x11, 0xc4, 0x8c },
		{ 0xaa, 0x01, 0x73, 0x9b, 0x8c, 0xaa, 0x97, 0x6d }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 3 */
		{ 0x9e, 0x59, 0x44, 0xae, 0xa9, 0x4b, 0x81, 0x16,
		  0x5c, 0x82, 0xfb, 0xf9, 0xf3, 0x2d, 0xb7, 0x51 },
		{ 0xce, 0x83, 0xdb, 0xc5, 0x4a, 0xc0, 0x27, 0x4a,
		  0x15, 0x7c, 0x17, 0xf8, 0x0d, 0x01, 0x7b, 0xd6 },
		{ 0xa6, 0x4a, 0x50, 0x7a, 0xe1, 0xa2, 0xa9, 0x8b,
		  0xb8, 0x8e, 0xb4, 0x21, 0x01, 0x35, 0xdc, 0x87 },
		{ 0xcf, 0xbc, 0xe3, 0xfe },
		{ 0xf3, 0x65, 0xcd, 0x68 },
		{ 0x9a, 0x8e, 0xc9, 0x5f, 0x40, 0x8c, 0xc5, 0x07 }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 4 */
		{ 0x4a, 0xb1, 0xde, 0xb0, 0x5c, 0xa6, 0xce, 0xb0,
		  0x51, 0xfc, 0x98, 0xe7, 0x7d, 0x02, 0x6a, 0x84 },
		{ 0x74, 0xb0, 0xcd, 0x60, 0x31, 0xa1, 0xc8, 0x33,
		  0x9b, 0x2b, 0x6c, 0xe2, 0xb8, 0xc4, 0xa1, 0x86 },
		{ 0xdc, 0xf0, 0x7c, 0xbd, 0x51, 0x85, 0x52, 0x90,
		  0xb9, 0x2a, 0x07, 0xa9, 0x89, 0x1e, 0x52, 0x3e },
		{ 0x96, 0x55, 0xe2, 0x65 },
		{ 0x58, 0x60, 0xfc, 0x1b },
		{ 0xcd, 0xc1, 0xdc, 0x08, 0x41, 0xb8, 0x1a, 0x22 }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 5 */
		{ 0x6c, 0x38, 0xa1, 0x16, 0xac, 0x28, 0x0c, 0x45,
		  0x4f, 0x59, 0x33, 0x2e, 0xe3, 0x5c, 0x8c, 0x4f },
		{ 0xee, 0x64, 0x66, 0xbc, 0x96, 0x20, 0x2c, 0x5a,
		  0x55, 0x7a, 0xbb, 0xef, 0xf8, 0xba, 0xbf, 0x63 },
		{ 0x38, 0x03, 0xef, 0x53, 0x63, 0xb9, 0x47, 0xc6,
		  0xaa, 0xa2, 0x25, 0xe5, 0x8f, 0xae, 0x39, 0x34 },
		{ 0x13, 0x68, 0x8f, 0x17 },
		{ 0x16, 0xc8, 0x23, 0x3f },
		{ 0xdf, 0x75, 0xbc, 0x5e, 0xa8, 0x99, 0x87, 0x9f }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 6 */
		{ 0x2d, 0x60, 0x9d, 0x4d, 0xb0, 0xac, 0x5b, 0xf0,
		  0xd2, 0xc0, 0xde, 0x26, 0x70, 0x14, 0xde, 0x0d },
		{ 0x19, 0x4a, 0xa7, 0x56, 0x01, 0x38, 0x96, 0xb7,
		  0x4b, 0x4a, 0x2a, 0x3b, 0x0a, 0xf4, 0x53, 0x9e },
		{ 0xc3, 0x5a, 0x0a, 0xb0, 0xbc, 0xbf, 0xc9, 0x25,
		  0x2c, 0xaf, 0xf1, 0x5f, 0x24, 0xef, 0xbd, 0xe0 },
		{ 0x55, 0x3d, 0x00, 0xb3 },
		{ 0x8c, 0x25, 0xa1, 0x6c },
		{ 0x84, 0xb4, 0x17, 0xae, 0x3a, 0xea, 0xb4, 0xf3 }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 7 */
		{ 0xa5, 0x30, 0xa7, 0xfe, 0x42, 0x8f, 0xad, 0x10,
		  0x82, 0xc4, 0x5e, 0xdd, 0xfc, 0xe1, 0x38, 0x84 },
		{ 0x3a, 0x4c, 0x2b, 0x32, 0x45, 0xc5, 0x0e, 0xb5,
		  0xc7, 0x1d, 0x08, 0x63, 0x93, 0x95, 0x76, 0x4d },
		{ 0x27, 0x95, 0x3e, 0x49, 0xbc, 0x8a, 0xf6, 0xdc,
		  0xc6, 0xe7, 0x30, 0xeb, 0x80, 0x28, 0x6b, 0xe3 },
		{ 0x59, 0xf1, 0xa4, 0x4a },
		{ 0xa6, 0x32, 0x41, 0xe1 },
		{ 0x3b, 0x4e, 0x24, 0x4c, 0xdc, 0x60, 0xce, 0x03 }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 8 */
		{ 0xd9, 0x15, 0x1c, 0xf0, 0x48, 0x96, 0xe2, 0x58,
		  0x30, 0xbf, 0x2e, 0x08, 0x26, 0x7b, 0x83, 0x60 },
		{ 0xf7, 0x61, 0xe5, 0xe9, 0x3d, 0x60, 0x3f, 0xeb,
		  0x73, 0x0e, 0x27, 0x55, 0x6c, 0xb8, 0xa2, 0xca },
		{ 0xc4, 0xc9, 0x3e, 0xff, 0xe8, 0xa0, 0x81, 0x38,
		  0xc2, 0x03, 0xd4, 0xc2, 0x7c, 0xe4, 0xe3, 0xd9 },
		{ 0x50, 0x58, 0x88, 0x61 },
		{ 0x4a, 0x90, 0xb2, 0x17 },
		{ 0x8d, 0x4e, 0xc0, 0x1d, 0xe5, 0x97, 0xac, 0xfe }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 9 */
		{ 0xa0, 0xe2, 0x97, 0x1b, 0x68, 0x22, 0xe8, 0xd3,
		  0x54, 0xa1, 0x8c, 0xc2, 0x35, 0x62, 0x4e, 0xcb },
		{ 0x08, 0xef, 0xf8, 0x28, 0xb1, 0x3f, 0xdb, 0x56,
		  0x27, 0x22, 0xc6, 0x5c, 0x7f, 0x30, 0xa9, 0xb2 },
		{ 0x82, 0xa2, 0x6f, 0x22, 0xbb, 0xa9, 0xe9, 0x48,
		  0x8f, 0x94, 0x9a, 0x10, 0xd9, 0x8e, 0x9c, 0xc4 },
		{ 0xcd, 0xe6, 0xb0, 0x27 },
		{ 0x4b, 0xc2, 0x21, 0x2d },
		{ 0xd8, 0xde, 0xbc, 0x4f, 0xfb, 0xcd, 0x60, 0xaa }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 10 */
		{ 0x0d, 0xa6, 0xf7, 0xba, 0x86, 0xd5, 0xea, 0xc8,
		  0xa1, 0x9c, 0xf5, 0x63, 0xac, 0x58, 0x64, 0x2d },
		{ 0x67, 0x9a, 0xc4, 0xdb, 0xac, 0xd7, 0xd2, 0x33,
		  0xff, 0x9d, 0x68, 0x06, 0xf4, 0x14, 0x9c, 0xe3 },
		{ 0x0d, 0xb1, 0x07, 0x1f, 0x87, 0x67, 0x56, 0x2c,
		  0xa4, 0x3a, 0x0a, 0x64, 0xc4, 0x1e, 0x8d, 0x08 },
		{ 0x02, 0xd1, 0x3a, 0xcd },
		{ 0x6f, 0xc3, 0x0f, 0xee },
		{ 0xf0, 0xea, 0xa5, 0x0a, 0x1e, 0xdc, 0xeb, 0xb7 }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 11 */
		{ 0x77, 0xb4, 0x58, 0x43, 0xc8, 0x8e, 0x58, 0xc1,
		  0x0d, 0x20, 0x26, 0x84, 0x51, 0x5e, 0xd4, 0x30 },
		{ 0x4c, 0x47, 0xeb, 0x30, 0x76, 0xdc, 0x55, 0xfe,
		  0x51, 0x06, 0xcb, 0x20, 0x34, 0xb8, 0xcd, 0x78 },
		{ 0xd4, 0x83, 0xaf, 0xae, 0x56, 0x24, 0x09, 0xa3,
		  0x26, 0xb5, 0xbb, 0x0b, 0x20, 0xc4, 0xd7, 0x62 },
		{ 0x44, 0x38, 0x9d, 0x01 },
		{ 0xae, 0xfa, 0x35, 0x7b },
		{ 0x82, 0xdb, 0xab, 0x7f, 0x83, 0xf0, 0x63, 0xda }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 12 */
		{ 0x72, 0x9b, 0x17, 0x72, 0x92, 0x70, 0xdd, 0x87,
		  0xcc, 0xdf, 0x1b, 0xfe, 0x29, 0xb4, 0xe9, 0xbb },
		{ 0x31, 0x1c, 0x4c, 0x92, 0x97, 0x44, 0xd6, 0x75,
		  0xb7, 0x20, 0xf3, 0xb7, 0xe9, 0xb1, 0xcb, 0xd0 },
		{ 0x22, 0x8c, 0x2f, 0x2f, 0x06, 0xac, 0x32, 0x68,
		  0xa9, 0xe6, 0x16, 0xee, 0x16, 0xdb, 0x4b, 0xa1 },
		{ 0x03, 0xe0, 0xfd, 0x84 },
		{ 0x98, 0xdb, 0xbd, 0x09 },
		{ 0x3c, 0x66, 0xcb, 0x98, 0xca, 0xb2, 0xd3, 0x3d }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 13 */
		{ 0xd3, 0x2d, 0xd2, 0x3e, 0x89, 0xdc, 0x66, 0x23,
		  0x54, 0xca, 0x12, 0xeb, 0x79, 0xdd, 0x32, 0xfa },
		{ 0xcf, 0x7d, 0x0a, 0xb1, 0xd9, 0x43, 0x06, 0x95,
		  0x0b, 0xf1, 0x20, 0x18, 0xfb, 0xd4, 0x68, 0x87 },
		{ 0xd2, 0x2a, 0x4b, 0x41, 0x80, 0xa5, 0x32, 0x57,
		  0x08, 0xa5, 0xff, 0x70, 0xd9, 0xf6, 0x7e, 0xc7 },
		{ 0xbe, 0x73, 0xb3, 0xdc },
		{ 0xaf, 0x4a, 0x41, 0x1e },
		{ 0x96, 0x12, 0xb5, 0xd8, 0x8a, 0x41, 0x30, 0xbb }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 14 */
		{ 0xaf, 0x7c, 0x65, 0xe1, 0x92, 0x72, 0x21, 0xde,
		  0x59, 0x11, 0x87, 0xa2, 0xc5, 0x98, 0x7a, 0x53 },
		{ 0x1f, 0x0f, 0x85, 0x78, 0x46, 0x4f, 0xd5, 0x9b,
		  0x64, 0xbe, 0xd2, 0xd0, 0x94, 0x36, 0xb5, 0x7a },
		{ 0xa4, 0xcf, 0x5c, 0x81, 0x55, 0xc0, 0x8a, 0x7e,
		  0xff, 0x41, 0x8e, 0x54, 0x43, 0xb9, 0x8e, 0x55 },
		{ 0x8f, 0xe0, 0x19, 0xc7 },
		{ 0x7b, 0xff, 0xa5, 0xc2 },
		{ 0x75, 0xa1, 0x50, 0xdf, 0x3c, 0x6a, 0xed, 0x08 }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 15 */
		{ 0x5b, 0xd7, 0xec, 0xd3, 0xd3, 0x12, 0x7a, 0x41,
		  0xd1, 0x25, 0x39, 0xbe, 0xd4, 0xe7, 0xcf, 0x71 },
		{ 0x59, 0xb7, 0x5f, 0x14, 0x25, 0x1c, 0x75, 0x03,
		  0x1d, 0x0b, 0xcb, 0xac, 0x1c, 0x2c, 0x04, 0xc7 },
		{ 0x76, 0x08, 0x9d, 0x3c, 0x0f, 0xf3, 0xef, 0xdc,
		  0x6e, 0x36, 0x72, 0x1d, 0x4f, 0xce, 0xb7, 0x47 },
		{ 0x27, 0x20, 0x2b, 0x82 },
		{ 0x7e, 0x3f, 0x44, 0xc7 },
		{ 0xb7, 0xf9, 0x2e, 0x42, 0x6a, 0x36, 0xfe, 0xc5 }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 16 */
		{ 0x6c, 0xd1, 0xc6, 0xce, 0xb1, 0xe0, 0x1e, 0x14,
		  0xf1, 0xb8, 0x23, 0x16, 0xa9, 0x0b, 0x7f, 0x3d },
		{ 0xf6, 0x9b, 0x78, 0xf3, 0x00, 0xa0, 0x56, 0x8b,
		  0xce, 0x9f, 0x0c, 0xb9, 0x3c, 0x4b, 0xe4, 0xc9 },
		{ 0xa2, 0x19, 0xdc, 0x37, 0xf1, 0xdc, 0x7d, 0x66,
		  0x73, 0x8b, 0x58, 0x43, 0xc7, 0x99, 0xf2, 0x06 },
		{ 0xdd, 0xd7, 0xef, 0xe6 },
		{ 0x70, 0xf6, 0xbd, 0xb9 },
		{ 0x88, 0xd9, 0xde, 0x10, 0xa2, 0x20, 0x04, 0xc5 }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 17 */
		{ 0xb7, 0x3a, 0x90, 0xcb, 0xcf, 0x3a, 0xfb, 0x62,
		  0x2d, 0xba, 0x83, 0xc5, 0x8a, 0x84, 0x15, 0xdf },
		{ 0xb1, 0x20, 0xf1, 0xc1, 0xa0, 0x10, 0x2a, 0x2f,
		  0x50, 0x7d, 0xd5, 0x43, 0xde, 0x68, 0x28, 0x1f },
		{ 0xdf, 0x0c, 0x67, 0x86, 0x8f, 0xa2, 0x5f, 0x74,
		  0x8b, 0x70, 0x44, 0xc6, 0xe7, 0xc2, 0x45, 0xb8 },
		{ 0x67, 0xe4, 0xff, 0x3f },
		{ 0x47, 0x9d, 0xd2, 0x5c },
		{ 0xa8, 0x19, 0xe5, 0x77, 0xa8, 0xd6, 0x17, 0x5b }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 18 */
		{ 0x51, 0x22, 0x25, 0x02, 0x14, 0xc3, 0x3e, 0x72,
		  0x3a, 0x5d, 0xd5, 0x23, 0xfc, 0x14, 0x5f, 0xc0 },
		{ 0x81, 0xe9, 0x2b, 0x6c, 0x0e, 0xe0, 0xe1, 0x2e,
		  0xbc, 0xeb, 0xa8, 0xd9, 0x2a, 0x99, 0xdf, 0xa5 },
		{ 0x98, 0x1d, 0x46, 0x4c, 0x7c, 0x52, 0xeb, 0x6e,
		  0x50, 0x36, 0x23, 0x49, 0x84, 0xad, 0x0b, 0xcf },
		{ 0x8a, 0x3b, 0x8d, 0x17 },
		{ 0x28, 0xd7, 0xb0, 0xf2 },
		{ 0x9a, 0x8d, 0x0e, 0x88, 0x3f, 0xf0, 0x88, 0x7a }
	}, {
		/* 3GPP TS 55.205 v6.0.0 - Test Set 19 */
		{ 0x90, 0xdc, 0xa4, 0xed, 0xa4, 0x5b, 0x53, 0xcf,
		  0x0f, 0x12, 0xd7, 0xc9, 0xc3, 0xbc, 0x6a, 0x89 },
		{ 0x9f, 0xdd, 0xc7, 0x20, 0x92, 0xc6, 0xad, 0x03,
		  0x6b, 0x6e, 0x46, 0x47, 0x89, 0x31, 0x5b, 0x78 },
		{ 0xcb, 0x9c, 0xcc, 0xc4, 0xb9, 0x25, 0x8e, 0x6d,
		  0xca, 0x47, 0x60, 0x37, 0x9f, 0xb8, 0x25, 0x81 },
		{ 0xdf, 0x58, 0x52, 0x2f },
		{ 0xa9, 0x51, 0x00, 0xe2 },
		{ 0xed, 0x29, 0xb2, 0xf1, 0xc2, 0x7f, 0x9f, 0x34 }
	}
};

#define NUM_GSM_TESTS (sizeof(gsm_test_sets) / sizeof(gsm_test_sets[0]))


struct milenage_test_set {
	u8 k[16];
	u8 rand[16];
	u8 sqn[6];
	u8 amf[2];
	u8 op[16];
	u8 opc[16];
	u8 f1[8];
	u8 f1star[8];
	u8 f2[8];
	u8 f3[16];
	u8 f4[16];
	u8 f5[6];
	u8 f5star[6];
};

static const struct milenage_test_set test_sets[] =
{
	{
		/* 3GPP TS 35.208 v6.0.0 - 4.3.1 Test Set 1 */
		{ 0x46, 0x5b, 0x5c, 0xe8, 0xb1, 0x99, 0xb4, 0x9f,
		  0xaa, 0x5f, 0x0a, 0x2e, 0xe2, 0x38, 0xa6, 0xbc },
		{ 0x23, 0x55, 0x3c, 0xbe, 0x96, 0x37, 0xa8, 0x9d,
		  0x21, 0x8a, 0xe6, 0x4d, 0xae, 0x47, 0xbf, 0x35 },
		{ 0xff, 0x9b, 0xb4, 0xd0, 0xb6, 0x07 },
		{ 0xb9, 0xb9 },
		{ 0xcd, 0xc2, 0x02, 0xd5, 0x12, 0x3e, 0x20, 0xf6,
		  0x2b, 0x6d, 0x67, 0x6a, 0xc7, 0x2c, 0xb3, 0x18 },
		{ 0xcd, 0x63, 0xcb, 0x71, 0x95, 0x4a, 0x9f, 0x4e,
		  0x48, 0xa5, 0x99, 0x4e, 0x37, 0xa0, 0x2b, 0xaf },
		{ 0x4a, 0x9f, 0xfa, 0xc3, 0x54, 0xdf, 0xaf, 0xb3 },
		{ 0x01, 0xcf, 0xaf, 0x9e, 0xc4, 0xe8, 0x71, 0xe9 },
		{ 0xa5, 0x42, 0x11, 0xd5, 0xe3, 0xba, 0x50, 0xbf },
		{ 0xb4, 0x0b, 0xa9, 0xa3, 0xc5, 0x8b, 0x2a, 0x05,
		  0xbb, 0xf0, 0xd9, 0x87, 0xb2, 0x1b, 0xf8, 0xcb },
		{ 0xf7, 0x69, 0xbc, 0xd7, 0x51, 0x04, 0x46, 0x04,
		  0x12, 0x76, 0x72, 0x71, 0x1c, 0x6d, 0x34, 0x41 },
		{ 0xaa, 0x68, 0x9c, 0x64, 0x83, 0x70 },
		{ 0x45, 0x1e, 0x8b, 0xec, 0xa4, 0x3b }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.2 Test Set 2 */
		{ 0x46, 0x5b, 0x5c, 0xe8, 0xb1, 0x99, 0xb4, 0x9f,
		  0xaa, 0x5f, 0x0a, 0x2e, 0xe2, 0x38, 0xa6, 0xbc },
		{ 0x23, 0x55, 0x3c, 0xbe, 0x96, 0x37, 0xa8, 0x9d,
		  0x21, 0x8a, 0xe6, 0x4d, 0xae, 0x47, 0xbf, 0x35 },
		{ 0xff, 0x9b, 0xb4, 0xd0, 0xb6, 0x07 },
		{ 0xb9, 0xb9 },
		{ 0xcd, 0xc2, 0x02, 0xd5, 0x12, 0x3e, 0x20, 0xf6,
		  0x2b, 0x6d, 0x67, 0x6a, 0xc7, 0x2c, 0xb3, 0x18 },
		{ 0xcd, 0x63, 0xcb, 0x71, 0x95, 0x4a, 0x9f, 0x4e,
		  0x48, 0xa5, 0x99, 0x4e, 0x37, 0xa0, 0x2b, 0xaf },
		{ 0x4a, 0x9f, 0xfa, 0xc3, 0x54, 0xdf, 0xaf, 0xb3 },
		{ 0x01, 0xcf, 0xaf, 0x9e, 0xc4, 0xe8, 0x71, 0xe9 },
		{ 0xa5, 0x42, 0x11, 0xd5, 0xe3, 0xba, 0x50, 0xbf },
		{ 0xb4, 0x0b, 0xa9, 0xa3, 0xc5, 0x8b, 0x2a, 0x05,
		  0xbb, 0xf0, 0xd9, 0x87, 0xb2, 0x1b, 0xf8, 0xcb },
		{ 0xf7, 0x69, 0xbc, 0xd7, 0x51, 0x04, 0x46, 0x04,
		  0x12, 0x76, 0x72, 0x71, 0x1c, 0x6d, 0x34, 0x41 },
		{ 0xaa, 0x68, 0x9c, 0x64, 0x83, 0x70 },
		{ 0x45, 0x1e, 0x8b, 0xec, 0xa4, 0x3b }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.3 Test Set 3 */
		{ 0xfe, 0xc8, 0x6b, 0xa6, 0xeb, 0x70, 0x7e, 0xd0,
		  0x89, 0x05, 0x75, 0x7b, 0x1b, 0xb4, 0x4b, 0x8f },
		{ 0x9f, 0x7c, 0x8d, 0x02, 0x1a, 0xcc, 0xf4, 0xdb,
		  0x21, 0x3c, 0xcf, 0xf0, 0xc7, 0xf7, 0x1a, 0x6a },
		{ 0x9d, 0x02, 0x77, 0x59, 0x5f, 0xfc },
		{ 0x72, 0x5c },
		{ 0xdb, 0xc5, 0x9a, 0xdc, 0xb6, 0xf9, 0xa0, 0xef,
		  0x73, 0x54, 0x77, 0xb7, 0xfa, 0xdf, 0x83, 0x74 },
		{ 0x10, 0x06, 0x02, 0x0f, 0x0a, 0x47, 0x8b, 0xf6,
		  0xb6, 0x99, 0xf1, 0x5c, 0x06, 0x2e, 0x42, 0xb3 },
		{ 0x9c, 0xab, 0xc3, 0xe9, 0x9b, 0xaf, 0x72, 0x81 },
		{ 0x95, 0x81, 0x4b, 0xa2, 0xb3, 0x04, 0x43, 0x24 },
		{ 0x80, 0x11, 0xc4, 0x8c, 0x0c, 0x21, 0x4e, 0xd2 },
		{ 0x5d, 0xbd, 0xbb, 0x29, 0x54, 0xe8, 0xf3, 0xcd,
		  0xe6, 0x65, 0xb0, 0x46, 0x17, 0x9a, 0x50, 0x98 },
		{ 0x59, 0xa9, 0x2d, 0x3b, 0x47, 0x6a, 0x04, 0x43,
		  0x48, 0x70, 0x55, 0xcf, 0x88, 0xb2, 0x30, 0x7b },
		{ 0x33, 0x48, 0x4d, 0xc2, 0x13, 0x6b },
		{ 0xde, 0xac, 0xdd, 0x84, 0x8c, 0xc6 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.4 Test Set 4 */
		{ 0x9e, 0x59, 0x44, 0xae, 0xa9, 0x4b, 0x81, 0x16,
		  0x5c, 0x82, 0xfb, 0xf9, 0xf3, 0x2d, 0xb7, 0x51 },
		{ 0xce, 0x83, 0xdb, 0xc5, 0x4a, 0xc0, 0x27, 0x4a,
		  0x15, 0x7c, 0x17, 0xf8, 0x0d, 0x01, 0x7b, 0xd6 },
		{ 0x0b, 0x60, 0x4a, 0x81, 0xec, 0xa8 },
		{ 0x9e, 0x09 },
		{ 0x22, 0x30, 0x14, 0xc5, 0x80, 0x66, 0x94, 0xc0,
		  0x07, 0xca, 0x1e, 0xee, 0xf5, 0x7f, 0x00, 0x4f },
		{ 0xa6, 0x4a, 0x50, 0x7a, 0xe1, 0xa2, 0xa9, 0x8b,
		  0xb8, 0x8e, 0xb4, 0x21, 0x01, 0x35, 0xdc, 0x87 },
		{ 0x74, 0xa5, 0x82, 0x20, 0xcb, 0xa8, 0x4c, 0x49 },
		{ 0xac, 0x2c, 0xc7, 0x4a, 0x96, 0x87, 0x18, 0x37 },
		{ 0xf3, 0x65, 0xcd, 0x68, 0x3c, 0xd9, 0x2e, 0x96 },
		{ 0xe2, 0x03, 0xed, 0xb3, 0x97, 0x15, 0x74, 0xf5,
		  0xa9, 0x4b, 0x0d, 0x61, 0xb8, 0x16, 0x34, 0x5d },
		{ 0x0c, 0x45, 0x24, 0xad, 0xea, 0xc0, 0x41, 0xc4,
		  0xdd, 0x83, 0x0d, 0x20, 0x85, 0x4f, 0xc4, 0x6b },
		{ 0xf0, 0xb9, 0xc0, 0x8a, 0xd0, 0x2e },
		{ 0x60, 0x85, 0xa8, 0x6c, 0x6f, 0x63 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.5 Test Set 5 */
		{ 0x4a, 0xb1, 0xde, 0xb0, 0x5c, 0xa6, 0xce, 0xb0,
		  0x51, 0xfc, 0x98, 0xe7, 0x7d, 0x02, 0x6a, 0x84 },
		{ 0x74, 0xb0, 0xcd, 0x60, 0x31, 0xa1, 0xc8, 0x33,
		  0x9b, 0x2b, 0x6c, 0xe2, 0xb8, 0xc4, 0xa1, 0x86 },
		{ 0xe8, 0x80, 0xa1, 0xb5, 0x80, 0xb6 },
		{ 0x9f, 0x07 },
		{ 0x2d, 0x16, 0xc5, 0xcd, 0x1f, 0xdf, 0x6b, 0x22,
		  0x38, 0x35, 0x84, 0xe3, 0xbe, 0xf2, 0xa8, 0xd8 },
		{ 0xdc, 0xf0, 0x7c, 0xbd, 0x51, 0x85, 0x52, 0x90,
		  0xb9, 0x2a, 0x07, 0xa9, 0x89, 0x1e, 0x52, 0x3e },
		{ 0x49, 0xe7, 0x85, 0xdd, 0x12, 0x62, 0x6e, 0xf2 },
		{ 0x9e, 0x85, 0x79, 0x03, 0x36, 0xbb, 0x3f, 0xa2 },
		{ 0x58, 0x60, 0xfc, 0x1b, 0xce, 0x35, 0x1e, 0x7e },
		{ 0x76, 0x57, 0x76, 0x6b, 0x37, 0x3d, 0x1c, 0x21,
		  0x38, 0xf3, 0x07, 0xe3, 0xde, 0x92, 0x42, 0xf9 },
		{ 0x1c, 0x42, 0xe9, 0x60, 0xd8, 0x9b, 0x8f, 0xa9,
		  0x9f, 0x27, 0x44, 0xe0, 0x70, 0x8c, 0xcb, 0x53 },
		{ 0x31, 0xe1, 0x1a, 0x60, 0x91, 0x18 },
		{ 0xfe, 0x25, 0x55, 0xe5, 0x4a, 0xa9 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.6 Test Set 6 */
		{ 0x6c, 0x38, 0xa1, 0x16, 0xac, 0x28, 0x0c, 0x45,
		  0x4f, 0x59, 0x33, 0x2e, 0xe3, 0x5c, 0x8c, 0x4f },
		{ 0xee, 0x64, 0x66, 0xbc, 0x96, 0x20, 0x2c, 0x5a,
		  0x55, 0x7a, 0xbb, 0xef, 0xf8, 0xba, 0xbf, 0x63 },
		{ 0x41, 0x4b, 0x98, 0x22, 0x21, 0x81 },
		{ 0x44, 0x64 },
		{ 0x1b, 0xa0, 0x0a, 0x1a, 0x7c, 0x67, 0x00, 0xac,
		  0x8c, 0x3f, 0xf3, 0xe9, 0x6a, 0xd0, 0x87, 0x25 },
		{ 0x38, 0x03, 0xef, 0x53, 0x63, 0xb9, 0x47, 0xc6,
		  0xaa, 0xa2, 0x25, 0xe5, 0x8f, 0xae, 0x39, 0x34 },
		{ 0x07, 0x8a, 0xdf, 0xb4, 0x88, 0x24, 0x1a, 0x57 },
		{ 0x80, 0x24, 0x6b, 0x8d, 0x01, 0x86, 0xbc, 0xf1 },
		{ 0x16, 0xc8, 0x23, 0x3f, 0x05, 0xa0, 0xac, 0x28 },
		{ 0x3f, 0x8c, 0x75, 0x87, 0xfe, 0x8e, 0x4b, 0x23,
		  0x3a, 0xf6, 0x76, 0xae, 0xde, 0x30, 0xba, 0x3b },
		{ 0xa7, 0x46, 0x6c, 0xc1, 0xe6, 0xb2, 0xa1, 0x33,
		  0x7d, 0x49, 0xd3, 0xb6, 0x6e, 0x95, 0xd7, 0xb4 },
		{ 0x45, 0xb0, 0xf6, 0x9a, 0xb0, 0x6c },
		{ 0x1f, 0x53, 0xcd, 0x2b, 0x11, 0x13 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.7 Test Set 7 */
		{ 0x2d, 0x60, 0x9d, 0x4d, 0xb0, 0xac, 0x5b, 0xf0,
		  0xd2, 0xc0, 0xde, 0x26, 0x70, 0x14, 0xde, 0x0d },
		{ 0x19, 0x4a, 0xa7, 0x56, 0x01, 0x38, 0x96, 0xb7,
		  0x4b, 0x4a, 0x2a, 0x3b, 0x0a, 0xf4, 0x53, 0x9e },
		{ 0x6b, 0xf6, 0x94, 0x38, 0xc2, 0xe4 },
		{ 0x5f, 0x67 },
		{ 0x46, 0x0a, 0x48, 0x38, 0x54, 0x27, 0xaa, 0x39,
		  0x26, 0x4a, 0xac, 0x8e, 0xfc, 0x9e, 0x73, 0xe8 },
		{ 0xc3, 0x5a, 0x0a, 0xb0, 0xbc, 0xbf, 0xc9, 0x25,
		  0x2c, 0xaf, 0xf1, 0x5f, 0x24, 0xef, 0xbd, 0xe0 },
		{ 0xbd, 0x07, 0xd3, 0x00, 0x3b, 0x9e, 0x5c, 0xc3 },
		{ 0xbc, 0xb6, 0xc2, 0xfc, 0xad, 0x15, 0x22, 0x50 },
		{ 0x8c, 0x25, 0xa1, 0x6c, 0xd9, 0x18, 0xa1, 0xdf },
		{ 0x4c, 0xd0, 0x84, 0x60, 0x20, 0xf8, 0xfa, 0x07,
		  0x31, 0xdd, 0x47, 0xcb, 0xdc, 0x6b, 0xe4, 0x11 },
		{ 0x88, 0xab, 0x80, 0xa4, 0x15, 0xf1, 0x5c, 0x73,
		  0x71, 0x12, 0x54, 0xa1, 0xd3, 0x88, 0xf6, 0x96 },
		{ 0x7e, 0x64, 0x55, 0xf3, 0x4c, 0xf3 },
		{ 0xdc, 0x6d, 0xd0, 0x1e, 0x8f, 0x15 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.8 Test Set 8 */
		{ 0xa5, 0x30, 0xa7, 0xfe, 0x42, 0x8f, 0xad, 0x10,
		  0x82, 0xc4, 0x5e, 0xdd, 0xfc, 0xe1, 0x38, 0x84 },
		{ 0x3a, 0x4c, 0x2b, 0x32, 0x45, 0xc5, 0x0e, 0xb5,
		  0xc7, 0x1d, 0x08, 0x63, 0x93, 0x95, 0x76, 0x4d },
		{ 0xf6, 0x3f, 0x5d, 0x76, 0x87, 0x84 },
		{ 0xb9, 0x0e },
		{ 0x51, 0x1c, 0x6c, 0x4e, 0x83, 0xe3, 0x8c, 0x89,
		  0xb1, 0xc5, 0xd8, 0xdd, 0xe6, 0x24, 0x26, 0xfa },
		{ 0x27, 0x95, 0x3e, 0x49, 0xbc, 0x8a, 0xf6, 0xdc,
		  0xc6, 0xe7, 0x30, 0xeb, 0x80, 0x28, 0x6b, 0xe3 },
		{ 0x53, 0x76, 0x1f, 0xbd, 0x67, 0x9b, 0x0b, 0xad },
		{ 0x21, 0xad, 0xfd, 0x33, 0x4a, 0x10, 0xe7, 0xce },
		{ 0xa6, 0x32, 0x41, 0xe1, 0xff, 0xc3, 0xe5, 0xab },
		{ 0x10, 0xf0, 0x5b, 0xab, 0x75, 0xa9, 0x9a, 0x5f,
		  0xbb, 0x98, 0xa9, 0xc2, 0x87, 0x67, 0x9c, 0x3b },
		{ 0xf9, 0xec, 0x08, 0x65, 0xeb, 0x32, 0xf2, 0x23,
		  0x69, 0xca, 0xde, 0x40, 0xc5, 0x9c, 0x3a, 0x44 },
		{ 0x88, 0x19, 0x6c, 0x47, 0x98, 0x6f },
		{ 0xc9, 0x87, 0xa3, 0xd2, 0x31, 0x15 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.9 Test Set 9 */
		{ 0xd9, 0x15, 0x1c, 0xf0, 0x48, 0x96, 0xe2, 0x58,
		  0x30, 0xbf, 0x2e, 0x08, 0x26, 0x7b, 0x83, 0x60 },
		{ 0xf7, 0x61, 0xe5, 0xe9, 0x3d, 0x60, 0x3f, 0xeb,
		  0x73, 0x0e, 0x27, 0x55, 0x6c, 0xb8, 0xa2, 0xca },
		{ 0x47, 0xee, 0x01, 0x99, 0x82, 0x0a },
		{ 0x91, 0x13 },
		{ 0x75, 0xfc, 0x22, 0x33, 0xa4, 0x42, 0x94, 0xee,
		  0x8e, 0x6d, 0xe2, 0x5c, 0x43, 0x53, 0xd2, 0x6b },
		{ 0xc4, 0xc9, 0x3e, 0xff, 0xe8, 0xa0, 0x81, 0x38,
		  0xc2, 0x03, 0xd4, 0xc2, 0x7c, 0xe4, 0xe3, 0xd9 },
		{ 0x66, 0xcc, 0x4b, 0xe4, 0x48, 0x62, 0xaf, 0x1f },
		{ 0x7a, 0x4b, 0x8d, 0x7a, 0x87, 0x53, 0xf2, 0x46 },
		{ 0x4a, 0x90, 0xb2, 0x17, 0x1a, 0xc8, 0x3a, 0x76 },
		{ 0x71, 0x23, 0x6b, 0x71, 0x29, 0xf9, 0xb2, 0x2a,
		  0xb7, 0x7e, 0xa7, 0xa5, 0x4c, 0x96, 0xda, 0x22 },
		{ 0x90, 0x52, 0x7e, 0xba, 0xa5, 0x58, 0x89, 0x68,
		  0xdb, 0x41, 0x72, 0x73, 0x25, 0xa0, 0x4d, 0x9e },
		{ 0x82, 0xa0, 0xf5, 0x28, 0x7a, 0x71 },
		{ 0x52, 0x7d, 0xbf, 0x41, 0xf3, 0x5f }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.10 Test Set 10 */
		{ 0xa0, 0xe2, 0x97, 0x1b, 0x68, 0x22, 0xe8, 0xd3,
		  0x54, 0xa1, 0x8c, 0xc2, 0x35, 0x62, 0x4e, 0xcb },
		{ 0x08, 0xef, 0xf8, 0x28, 0xb1, 0x3f, 0xdb, 0x56,
		  0x27, 0x22, 0xc6, 0x5c, 0x7f, 0x30, 0xa9, 0xb2 },
		{ 0xdb, 0x5c, 0x06, 0x64, 0x81, 0xe0 },
		{ 0x71, 0x6b },
		{ 0x32, 0x37, 0x92, 0xfa, 0xca, 0x21, 0xfb, 0x4d,
		  0x5d, 0x6f, 0x13, 0xc1, 0x45, 0xa9, 0xd2, 0xc1 },
		{ 0x82, 0xa2, 0x6f, 0x22, 0xbb, 0xa9, 0xe9, 0x48,
		  0x8f, 0x94, 0x9a, 0x10, 0xd9, 0x8e, 0x9c, 0xc4 },
		{ 0x94, 0x85, 0xfe, 0x24, 0x62, 0x1c, 0xb9, 0xf6 },
		{ 0xbc, 0xe3, 0x25, 0xce, 0x03, 0xe2, 0xe9, 0xb9 },
		{ 0x4b, 0xc2, 0x21, 0x2d, 0x86, 0x24, 0x91, 0x0a },
		{ 0x08, 0xce, 0xf6, 0xd0, 0x04, 0xec, 0x61, 0x47,
		  0x1a, 0x3c, 0x3c, 0xda, 0x04, 0x81, 0x37, 0xfa },
		{ 0xed, 0x03, 0x18, 0xca, 0x5d, 0xeb, 0x92, 0x06,
		  0x27, 0x2f, 0x6e, 0x8f, 0xa6, 0x4b, 0xa4, 0x11 },
		{ 0xa2, 0xf8, 0x58, 0xaa, 0x9e, 0x5d },
		{ 0x74, 0xe7, 0x6f, 0xbb, 0xec, 0x38 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.11 Test Set 11 */
		{ 0x0d, 0xa6, 0xf7, 0xba, 0x86, 0xd5, 0xea, 0xc8,
		  0xa1, 0x9c, 0xf5, 0x63, 0xac, 0x58, 0x64, 0x2d },
		{ 0x67, 0x9a, 0xc4, 0xdb, 0xac, 0xd7, 0xd2, 0x33,
		  0xff, 0x9d, 0x68, 0x06, 0xf4, 0x14, 0x9c, 0xe3 },
		{ 0x6e, 0x23, 0x31, 0xd6, 0x92, 0xad },
		{ 0x22, 0x4a },
		{ 0x4b, 0x9a, 0x26, 0xfa, 0x45, 0x9e, 0x3a, 0xcb,
		  0xff, 0x36, 0xf4, 0x01, 0x5d, 0xe3, 0xbd, 0xc1 },
		{ 0x0d, 0xb1, 0x07, 0x1f, 0x87, 0x67, 0x56, 0x2c,
		  0xa4, 0x3a, 0x0a, 0x64, 0xc4, 0x1e, 0x8d, 0x08 },
		{ 0x28, 0x31, 0xd7, 0xae, 0x90, 0x88, 0xe4, 0x92 },
		{ 0x9b, 0x2e, 0x16, 0x95, 0x11, 0x35, 0xd5, 0x23 },
		{ 0x6f, 0xc3, 0x0f, 0xee, 0x6d, 0x12, 0x35, 0x23 },
		{ 0x69, 0xb1, 0xca, 0xe7, 0xc7, 0x42, 0x9d, 0x97,
		  0x5e, 0x24, 0x5c, 0xac, 0xb0, 0x5a, 0x51, 0x7c },
		{ 0x74, 0xf2, 0x4e, 0x8c, 0x26, 0xdf, 0x58, 0xe1,
		  0xb3, 0x8d, 0x7d, 0xcd, 0x4f, 0x1b, 0x7f, 0xbd },
		{ 0x4c, 0x53, 0x9a, 0x26, 0xe1, 0xfa },
		{ 0x07, 0x86, 0x1e, 0x12, 0x69, 0x28 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.12 Test Set 12 */
		{ 0x77, 0xb4, 0x58, 0x43, 0xc8, 0x8e, 0x58, 0xc1,
		  0x0d, 0x20, 0x26, 0x84, 0x51, 0x5e, 0xd4, 0x30 },
		{ 0x4c, 0x47, 0xeb, 0x30, 0x76, 0xdc, 0x55, 0xfe,
		  0x51, 0x06, 0xcb, 0x20, 0x34, 0xb8, 0xcd, 0x78 },
		{ 0xfe, 0x1a, 0x87, 0x31, 0x00, 0x5d },
		{ 0xad, 0x25 },
		{ 0xbf, 0x32, 0x86, 0xc7, 0xa5, 0x14, 0x09, 0xce,
		  0x95, 0x72, 0x4d, 0x50, 0x3b, 0xfe, 0x6e, 0x70 },
		{ 0xd4, 0x83, 0xaf, 0xae, 0x56, 0x24, 0x09, 0xa3,
		  0x26, 0xb5, 0xbb, 0x0b, 0x20, 0xc4, 0xd7, 0x62 },
		{ 0x08, 0x33, 0x2d, 0x7e, 0x9f, 0x48, 0x45, 0x70 },
		{ 0xed, 0x41, 0xb7, 0x34, 0x48, 0x9d, 0x52, 0x07 },
		{ 0xae, 0xfa, 0x35, 0x7b, 0xea, 0xc2, 0xa8, 0x7a },
		{ 0x90, 0x8c, 0x43, 0xf0, 0x56, 0x9c, 0xb8, 0xf7,
		  0x4b, 0xc9, 0x71, 0xe7, 0x06, 0xc3, 0x6c, 0x5f },
		{ 0xc2, 0x51, 0xdf, 0x0d, 0x88, 0x8d, 0xd9, 0x32,
		  0x9b, 0xcf, 0x46, 0x65, 0x5b, 0x22, 0x6e, 0x40 },
		{ 0x30, 0xff, 0x25, 0xcd, 0xad, 0xf6 },
		{ 0xe8, 0x4e, 0xd0, 0xd4, 0x67, 0x7e }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.13 Test Set 13 */
		{ 0x72, 0x9b, 0x17, 0x72, 0x92, 0x70, 0xdd, 0x87,
		  0xcc, 0xdf, 0x1b, 0xfe, 0x29, 0xb4, 0xe9, 0xbb },
		{ 0x31, 0x1c, 0x4c, 0x92, 0x97, 0x44, 0xd6, 0x75,
		  0xb7, 0x20, 0xf3, 0xb7, 0xe9, 0xb1, 0xcb, 0xd0 },
		{ 0xc8, 0x5c, 0x4c, 0xf6, 0x59, 0x16 },
		{ 0x5b, 0xb2 },
		{ 0xd0, 0x4c, 0x9c, 0x35, 0xbd, 0x22, 0x62, 0xfa,
		  0x81, 0x0d, 0x29, 0x24, 0xd0, 0x36, 0xfd, 0x13 },
		{ 0x22, 0x8c, 0x2f, 0x2f, 0x06, 0xac, 0x32, 0x68,
		  0xa9, 0xe6, 0x16, 0xee, 0x16, 0xdb, 0x4b, 0xa1 },
		{ 0xff, 0x79, 0x4f, 0xe2, 0xf8, 0x27, 0xeb, 0xf8 },
		{ 0x24, 0xfe, 0x4d, 0xc6, 0x1e, 0x87, 0x4b, 0x52 },
		{ 0x98, 0xdb, 0xbd, 0x09, 0x9b, 0x3b, 0x40, 0x8d },
		{ 0x44, 0xc0, 0xf2, 0x3c, 0x54, 0x93, 0xcf, 0xd2,
		  0x41, 0xe4, 0x8f, 0x19, 0x7e, 0x1d, 0x10, 0x12 },
		{ 0x0c, 0x9f, 0xb8, 0x16, 0x13, 0x88, 0x4c, 0x25,
		  0x35, 0xdd, 0x0e, 0xab, 0xf3, 0xb4, 0x40, 0xd8 },
		{ 0x53, 0x80, 0xd1, 0x58, 0xcf, 0xe3 },
		{ 0x87, 0xac, 0x3b, 0x55, 0x9f, 0xb6 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.14 Test Set 14 */
		{ 0xd3, 0x2d, 0xd2, 0x3e, 0x89, 0xdc, 0x66, 0x23,
		  0x54, 0xca, 0x12, 0xeb, 0x79, 0xdd, 0x32, 0xfa },
		{ 0xcf, 0x7d, 0x0a, 0xb1, 0xd9, 0x43, 0x06, 0x95,
		  0x0b, 0xf1, 0x20, 0x18, 0xfb, 0xd4, 0x68, 0x87 },
		{ 0x48, 0x41, 0x07, 0xe5, 0x6a, 0x43 },
		{ 0xb5, 0xe6 },
		{ 0xfe, 0x75, 0x90, 0x5b, 0x9d, 0xa4, 0x7d, 0x35,
		  0x62, 0x36, 0xd0, 0x31, 0x4e, 0x09, 0xc3, 0x2e },
		{ 0xd2, 0x2a, 0x4b, 0x41, 0x80, 0xa5, 0x32, 0x57,
		  0x08, 0xa5, 0xff, 0x70, 0xd9, 0xf6, 0x7e, 0xc7 },
		{ 0xcf, 0x19, 0xd6, 0x2b, 0x6a, 0x80, 0x98, 0x66 },
		{ 0x5d, 0x26, 0x95, 0x37, 0xe4, 0x5e, 0x2c, 0xe6 },
		{ 0xaf, 0x4a, 0x41, 0x1e, 0x11, 0x39, 0xf2, 0xc2 },
		{ 0x5a, 0xf8, 0x6b, 0x80, 0xed, 0xb7, 0x0d, 0xf5,
		  0x29, 0x2c, 0xc1, 0x12, 0x1c, 0xba, 0xd5, 0x0c },
		{ 0x7f, 0x4d, 0x6a, 0xe7, 0x44, 0x0e, 0x18, 0x78,
		  0x9a, 0x8b, 0x75, 0xad, 0x3f, 0x42, 0xf0, 0x3a },
		{ 0x21, 0x7a, 0xf4, 0x92, 0x72, 0xad },
		{ 0x90, 0x0e, 0x10, 0x1c, 0x67, 0x7e }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.15 Test Set 15 */
		{ 0xaf, 0x7c, 0x65, 0xe1, 0x92, 0x72, 0x21, 0xde,
		  0x59, 0x11, 0x87, 0xa2, 0xc5, 0x98, 0x7a, 0x53 },
		{ 0x1f, 0x0f, 0x85, 0x78, 0x46, 0x4f, 0xd5, 0x9b,
		  0x64, 0xbe, 0xd2, 0xd0, 0x94, 0x36, 0xb5, 0x7a },
		{ 0x3d, 0x62, 0x7b, 0x01, 0x41, 0x8d },
		{ 0x84, 0xf6 },
		{ 0x0c, 0x7a, 0xcb, 0x8d, 0x95, 0xb7, 0xd4, 0xa3,
		  0x1c, 0x5a, 0xca, 0x6d, 0x26, 0x34, 0x5a, 0x88 },
		{ 0xa4, 0xcf, 0x5c, 0x81, 0x55, 0xc0, 0x8a, 0x7e,
		  0xff, 0x41, 0x8e, 0x54, 0x43, 0xb9, 0x8e, 0x55 },
		{ 0xc3, 0x7c, 0xae, 0x78, 0x05, 0x64, 0x20, 0x32 },
		{ 0x68, 0xcd, 0x09, 0xa4, 0x52, 0xd8, 0xdb, 0x7c },
		{ 0x7b, 0xff, 0xa5, 0xc2, 0xf4, 0x1f, 0xbc, 0x05 },
		{ 0x3f, 0x8c, 0x3f, 0x3c, 0xcf, 0x76, 0x25, 0xbf,
		  0x77, 0xfc, 0x94, 0xbc, 0xfd, 0x22, 0xfd, 0x26 },
		{ 0xab, 0xcb, 0xae, 0x8f, 0xd4, 0x61, 0x15, 0xe9,
		  0x96, 0x1a, 0x55, 0xd0, 0xda, 0x5f, 0x20, 0x78 },
		{ 0x83, 0x7f, 0xd7, 0xb7, 0x44, 0x19 },
		{ 0x56, 0xe9, 0x7a, 0x60, 0x90, 0xb1 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.16 Test Set 16 */
		{ 0x5b, 0xd7, 0xec, 0xd3, 0xd3, 0x12, 0x7a, 0x41,
		  0xd1, 0x25, 0x39, 0xbe, 0xd4, 0xe7, 0xcf, 0x71 },
		{ 0x59, 0xb7, 0x5f, 0x14, 0x25, 0x1c, 0x75, 0x03,
		  0x1d, 0x0b, 0xcb, 0xac, 0x1c, 0x2c, 0x04, 0xc7 },
		{ 0xa2, 0x98, 0xae, 0x89, 0x29, 0xdc },
		{ 0xd0, 0x56 },
		{ 0xf9, 0x67, 0xf7, 0x60, 0x38, 0xb9, 0x20, 0xa9,
		  0xcd, 0x25, 0xe1, 0x0c, 0x08, 0xb4, 0x99, 0x24 },
		{ 0x76, 0x08, 0x9d, 0x3c, 0x0f, 0xf3, 0xef, 0xdc,
		  0x6e, 0x36, 0x72, 0x1d, 0x4f, 0xce, 0xb7, 0x47 },
		{ 0xc3, 0xf2, 0x5c, 0xd9, 0x43, 0x09, 0x10, 0x7e },
		{ 0xb0, 0xc8, 0xba, 0x34, 0x36, 0x65, 0xaf, 0xcc },
		{ 0x7e, 0x3f, 0x44, 0xc7, 0x59, 0x1f, 0x6f, 0x45 },
		{ 0xd4, 0x2b, 0x2d, 0x61, 0x5e, 0x49, 0xa0, 0x3a,
		  0xc2, 0x75, 0xa5, 0xae, 0xf9, 0x7a, 0xf8, 0x92 },
		{ 0x0b, 0x3f, 0x8d, 0x02, 0x4f, 0xe6, 0xbf, 0xaf,
		  0xaa, 0x98, 0x2b, 0x8f, 0x82, 0xe3, 0x19, 0xc2 },
		{ 0x5b, 0xe1, 0x14, 0x95, 0x52, 0x5d },
		{ 0x4d, 0x6a, 0x34, 0xa1, 0xe4, 0xeb }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.17 Test Set 17 */
		{ 0x6c, 0xd1, 0xc6, 0xce, 0xb1, 0xe0, 0x1e, 0x14,
		  0xf1, 0xb8, 0x23, 0x16, 0xa9, 0x0b, 0x7f, 0x3d },
		{ 0xf6, 0x9b, 0x78, 0xf3, 0x00, 0xa0, 0x56, 0x8b,
		  0xce, 0x9f, 0x0c, 0xb9, 0x3c, 0x4b, 0xe4, 0xc9 },
		{ 0xb4, 0xfc, 0xe5, 0xfe, 0xb0, 0x59 },
		{ 0xe4, 0xbb },
		{ 0x07, 0x8b, 0xfc, 0xa9, 0x56, 0x46, 0x59, 0xec,
		  0xd8, 0x85, 0x1e, 0x84, 0xe6, 0xc5, 0x9b, 0x48 },
		{ 0xa2, 0x19, 0xdc, 0x37, 0xf1, 0xdc, 0x7d, 0x66,
		  0x73, 0x8b, 0x58, 0x43, 0xc7, 0x99, 0xf2, 0x06 },
		{ 0x69, 0xa9, 0x08, 0x69, 0xc2, 0x68, 0xcb, 0x7b },
		{ 0x2e, 0x0f, 0xdc, 0xf9, 0xfd, 0x1c, 0xfa, 0x6a },
		{ 0x70, 0xf6, 0xbd, 0xb9, 0xad, 0x21, 0x52, 0x5f },
		{ 0x6e, 0xda, 0xf9, 0x9e, 0x5b, 0xd9, 0xf8, 0x5d,
		  0x5f, 0x36, 0xd9, 0x1c, 0x12, 0x72, 0xfb, 0x4b },
		{ 0xd6, 0x1c, 0x85, 0x3c, 0x28, 0x0d, 0xd9, 0xc4,
		  0x6f, 0x29, 0x7b, 0xae, 0xc3, 0x86, 0xde, 0x17 },
		{ 0x1c, 0x40, 0x8a, 0x85, 0x8b, 0x3e },
		{ 0xaa, 0x4a, 0xe5, 0x2d, 0xaa, 0x30 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.18 Test Set 18 */
		{ 0xb7, 0x3a, 0x90, 0xcb, 0xcf, 0x3a, 0xfb, 0x62,
		  0x2d, 0xba, 0x83, 0xc5, 0x8a, 0x84, 0x15, 0xdf },
		{ 0xb1, 0x20, 0xf1, 0xc1, 0xa0, 0x10, 0x2a, 0x2f,
		  0x50, 0x7d, 0xd5, 0x43, 0xde, 0x68, 0x28, 0x1f },
		{ 0xf1, 0xe8, 0xa5, 0x23, 0xa3, 0x6d },
		{ 0x47, 0x1b },
		{ 0xb6, 0x72, 0x04, 0x7e, 0x00, 0x3b, 0xb9, 0x52,
		  0xdc, 0xa6, 0xcb, 0x8a, 0xf0, 0xe5, 0xb7, 0x79 },
		{ 0xdf, 0x0c, 0x67, 0x86, 0x8f, 0xa2, 0x5f, 0x74,
		  0x8b, 0x70, 0x44, 0xc6, 0xe7, 0xc2, 0x45, 0xb8 },
		{ 0xeb, 0xd7, 0x03, 0x41, 0xbc, 0xd4, 0x15, 0xb0 },
		{ 0x12, 0x35, 0x9f, 0x5d, 0x82, 0x22, 0x0c, 0x14 },
		{ 0x47, 0x9d, 0xd2, 0x5c, 0x20, 0x79, 0x2d, 0x63 },
		{ 0x66, 0x19, 0x5d, 0xbe, 0xd0, 0x31, 0x32, 0x74,
		  0xc5, 0xca, 0x77, 0x66, 0x61, 0x5f, 0xa2, 0x5e },
		{ 0x66, 0xbe, 0xc7, 0x07, 0xeb, 0x2a, 0xfc, 0x47,
		  0x6d, 0x74, 0x08, 0xa8, 0xf2, 0x92, 0x7b, 0x36 },
		{ 0xae, 0xfd, 0xaa, 0x5d, 0xdd, 0x99 },
		{ 0x12, 0xec, 0x2b, 0x87, 0xfb, 0xb1 }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.19 Test Set 19 */
		{ 0x51, 0x22, 0x25, 0x02, 0x14, 0xc3, 0x3e, 0x72,
		  0x3a, 0x5d, 0xd5, 0x23, 0xfc, 0x14, 0x5f, 0xc0 },
		{ 0x81, 0xe9, 0x2b, 0x6c, 0x0e, 0xe0, 0xe1, 0x2e,
		  0xbc, 0xeb, 0xa8, 0xd9, 0x2a, 0x99, 0xdf, 0xa5 },
		{ 0x16, 0xf3, 0xb3, 0xf7, 0x0f, 0xc2 },
		{ 0xc3, 0xab },
		{ 0xc9, 0xe8, 0x76, 0x32, 0x86, 0xb5, 0xb9, 0xff,
		  0xbd, 0xf5, 0x6e, 0x12, 0x97, 0xd0, 0x88, 0x7b },
		{ 0x98, 0x1d, 0x46, 0x4c, 0x7c, 0x52, 0xeb, 0x6e,
		  0x50, 0x36, 0x23, 0x49, 0x84, 0xad, 0x0b, 0xcf },
		{ 0x2a, 0x5c, 0x23, 0xd1, 0x5e, 0xe3, 0x51, 0xd5 },
		{ 0x62, 0xda, 0xe3, 0x85, 0x3f, 0x3a, 0xf9, 0xd2 },
		{ 0x28, 0xd7, 0xb0, 0xf2, 0xa2, 0xec, 0x3d, 0xe5 },
		{ 0x53, 0x49, 0xfb, 0xe0, 0x98, 0x64, 0x9f, 0x94,
		  0x8f, 0x5d, 0x2e, 0x97, 0x3a, 0x81, 0xc0, 0x0f },
		{ 0x97, 0x44, 0x87, 0x1a, 0xd3, 0x2b, 0xf9, 0xbb,
		  0xd1, 0xdd, 0x5c, 0xe5, 0x4e, 0x3e, 0x2e, 0x5a },
		{ 0xad, 0xa1, 0x5a, 0xeb, 0x7b, 0xb8 },
		{ 0xd4, 0x61, 0xbc, 0x15, 0x47, 0x5d }
	}, {
		/* 3GPP TS 35.208 v6.0.0 - 4.3.20 Test Set 20 */
		{ 0x90, 0xdc, 0xa4, 0xed, 0xa4, 0x5b, 0x53, 0xcf,
		  0x0f, 0x12, 0xd7, 0xc9, 0xc3, 0xbc, 0x6a, 0x89 },
		{ 0x9f, 0xdd, 0xc7, 0x20, 0x92, 0xc6, 0xad, 0x03,
		  0x6b, 0x6e, 0x46, 0x47, 0x89, 0x31, 0x5b, 0x78 },
		{ 0x20, 0xf8, 0x13, 0xbd, 0x41, 0x41 },
		{ 0x61, 0xdf },
		{ 0x3f, 0xfc, 0xfe, 0x5b, 0x7b, 0x11, 0x11, 0x58,
		  0x99, 0x20, 0xd3, 0x52, 0x8e, 0x84, 0xe6, 0x55 },
		{ 0xcb, 0x9c, 0xcc, 0xc4, 0xb9, 0x25, 0x8e, 0x6d,
		  0xca, 0x47, 0x60, 0x37, 0x9f, 0xb8, 0x25, 0x81 },
		{ 0x09, 0xdb, 0x94, 0xea, 0xb4, 0xf8, 0x14, 0x9e },
		{ 0xa2, 0x94, 0x68, 0xaa, 0x97, 0x75, 0xb5, 0x27 },
		{ 0xa9, 0x51, 0x00, 0xe2, 0x76, 0x09, 0x52, 0xcd },
		{ 0xb5, 0xf2, 0xda, 0x03, 0x88, 0x3b, 0x69, 0xf9,
		  0x6b, 0xf5, 0x2e, 0x02, 0x9e, 0xd9, 0xac, 0x45 },
		{ 0xb4, 0x72, 0x13, 0x68, 0xbc, 0x16, 0xea, 0x67,
		  0x87, 0x5c, 0x55, 0x98, 0x68, 0x8b, 0xb0, 0xef },
		{ 0x83, 0xcf, 0xd5, 0x4d, 0xb9, 0x13 },
		{ 0x4f, 0x20, 0x39, 0x39, 0x2d, 0xdc }
	}
};

#define NUM_TESTS (sizeof(test_sets) / sizeof(test_sets[0]))


int main(int argc, char *argv[])
{
	u8 buf[16], buf2[16], buf3[16], buf4[16], buf5[16], opc[16];
	u8 auts[14], sqn[6], _rand[16];
	int ret = 0, res, i;
	const struct milenage_test_set *t;
	size_t res_len;

	wpa_debug_level = 0;

	printf("Milenage test sets\n");
	for (i = 0; i < NUM_TESTS; i++) {
		t = &test_sets[i];
		printf("Test Set %d\n", i + 1);

		milenage_opc(t->op, t->k, opc);
		if (memcmp(opc, t->opc, 16) != 0) {
			printf("- milenage_opc failed\n");
			ret++;
		}

		if (milenage_f1(opc, t->k, t->rand, t->sqn, t->amf, buf, buf2)
		    ||  memcmp(buf, t->f1, 8) != 0) {
			printf("- milenage_f1 failed\n");
			ret++;
		}
		if (memcmp(buf2, t->f1star, 8) != 0) {
			printf("- milenage_f1* failed\n");
			ret++;
		}

		if (milenage_f2345(opc, t->k, t->rand, buf, buf2, buf3, buf4,
				   buf5) ||
		    memcmp(buf, t->f2, 8) != 0) {
			printf("- milenage_f2 failed\n");
			ret++;
		}
		if (memcmp(buf2, t->f3, 16) != 0) {
			printf("- milenage_f3 failed\n");
			ret++;
		}
		if (memcmp(buf3, t->f4, 16) != 0) {
			printf("- milenage_f4 failed\n");
			ret++;
		}
		if (memcmp(buf4, t->f5, 6) != 0) {
			printf("- milenage_f5 failed\n");
			ret++;
		}
		if (memcmp(buf5, t->f5star, 6) != 0) {
			printf("- milenage_f5* failed\n");
			ret++;
		}
	}

	printf("milenage_auts test:\n");
	os_memcpy(auts, "\x4f\x20\x39\x39\x2d\xdd", 6);
	os_memcpy(auts + 6, "\x4b\xb4\x31\x6e\xd4\xa1\x46\x88", 8);
	res = milenage_auts(t->opc, t->k, t->rand, auts, buf);
	printf("AUTS for test set %d: %d / SQN=%02x%02x%02x%02x%02x%02x\n",
	       i, res, buf[0], buf[1], buf[2], buf[3], buf[4], buf[5]);
	if (res)
		ret++;

	os_memset(_rand, 0xaa, sizeof(_rand));
	os_memcpy(auts,
		  "\x43\x68\x1a\xd3\xda\xf0\x06\xbc\xde\x40\x5a\x20\x72\x67",
		  14);
	res = milenage_auts(t->opc, t->k, _rand, auts, buf);
	printf("AUTS from a test USIM: %d / SQN=%02x%02x%02x%02x%02x%02x\n",
	       res, buf[0], buf[1], buf[2], buf[3], buf[4], buf[5]);
	if (res)
		ret++;

	printf("milenage_generate test:\n");
	os_memcpy(sqn, "\x00\x00\x00\x00\x40\x44", 6);
	os_memcpy(_rand, "\x12\x69\xb8\x23\x41\x39\x35\x66\xfb\x99\x41\xe9\x84"
		  "\x4f\xe6\x2f", 16);
	res_len = 8;
	milenage_generate(t->opc, t->amf, t->k, sqn, _rand, buf, buf2, buf3,
			  buf4, &res_len);
	wpa_hexdump(MSG_DEBUG, "SQN", sqn, 6);
	wpa_hexdump(MSG_DEBUG, "RAND", _rand, 16);
	wpa_hexdump(MSG_DEBUG, "AUTN", buf, 16);
	wpa_hexdump(MSG_DEBUG, "IK", buf2, 16);
	wpa_hexdump(MSG_DEBUG, "CK", buf3, 16);
	wpa_hexdump(MSG_DEBUG, "RES", buf4, res_len);

	printf("GSM-Milenage test sets\n");
	for (i = 0; i < NUM_GSM_TESTS; i++) {
		const struct gsm_milenage_test_set *g;
		u8 sres[4], kc[8];
		g = &gsm_test_sets[i];
		printf("Test Set %d\n", i + 1);
		gsm_milenage(g->opc, g->ki, g->rand, sres, kc);
		if (memcmp(g->kc, kc, 8) != 0) {
			printf("- gsm_milenage Kc failed\n");
			ret++;
		}
#ifdef GSM_MILENAGE_ALT_SRES
		if (memcmp(g->sres2, sres, 4) != 0) {
			printf("- gsm_milenage SRES#2 failed\n");
			ret++;
		}
#else /* GSM_MILENAGE_ALT_SRES */
		if (memcmp(g->sres1, sres, 4) != 0) {
			printf("- gsm_milenage SRES#1 failed\n");
			ret++;
		}
#endif /* GSM_MILENAGE_ALT_SRES */
	}

	if (ret)
		printf("Something failed\n");
	else
		printf("OK\n");

	return ret;
}
#endif /* TEST_MAIN_MILENAGE */
