/**
 * @file config.h
 * @author Radek Krejci <rkrejci@cesnet.cz>
 * @brief Various variables provided by cmake and compile time options.
 *
 * Copyright (c) 2021 CESNET, z.s.p.o.
 *
 * This source code is licensed under BSD 3-Clause License (the "License").
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://opensource.org/licenses/BSD-3-Clause
 */

#ifndef LY_CONFIG_H_
#define LY_CONFIG_H_

#ifdef _WIN32
/* headers are broken on Windows, which means that some of them simply *have* to come first */
# include <winsock2.h>
# include <ws2tcpip.h>
#endif

/** size of fixed_mem in lyd_value, minimum is 8 (B) */
#define LYD_VALUE_FIXED_MEM_SIZE 24

/** plugins */
#define LYPLG_SUFFIX ".so"
#define LYPLG_SUFFIX_LEN (sizeof LYPLG_SUFFIX - 1)
#define LYPLG_TYPE_DIR "/usr/lib/libyang/types"
#define LYPLG_EXT_DIR "/usr/lib/libyang/extensions"

/** atomic compiler operations, to be able to use uint32_t */
#ifndef _WIN32
# define LY_ATOMIC_INC_BARRIER(var) __sync_fetch_and_add(&(var), 1)
# define LY_ATOMIC_DEC_BARRIER(var) __sync_fetch_and_sub(&(var), 1)
#else
#  include <windows.h>
# define LY_ATOMIC_INC_BARRIER(var) InterlockedExchangeAdd(&(var), 1)
# define LY_ATOMIC_DEC_BARRIER(var) InterlockedExchangeAdd(&(var), -1)
#endif

/** printf compiler attribute */
#ifdef __GNUC__
# define _FORMAT_PRINTF(FORM, ARGS) __attribute__((format (printf, FORM, ARGS)))
#else
# define _FORMAT_PRINTF(FORM, ARGS)
#endif

/** Exporting symbols to a shared library and importing back afterwards
 *
 * - use LIBYANG_API_DECL to mark a declaration in the public header
 * - use LIBYANG_API_DEF to mark a definition (in the source code for the actual implementaiton)
 * */
#ifdef _MSC_VER
#  ifndef STATIC
#    define LIBYANG_API_DEF __declspec(dllexport)
#    ifdef LIBYANG_BUILD
#      define LIBYANG_API_DECL __declspec(dllexport)
#    else
#      define LIBYANG_API_DECL __declspec(dllimport)
#    endif
#  endif
#else
/*
 * If the compiler supports attribute to mark objects as hidden, mark all
 * objects as hidden and export only objects explicitly marked to be part of
 * the public API.
 */
#  define LIBYANG_API_DEF __attribute__((visibility("default")))
#  define LIBYANG_API_DECL
#endif

#endif /* LY_CONFIG_H_ */
