#pragma once

#include <map>
#include <memory>
#include <string>
#include <vector>

#include "usl_gpio.h"


class USL_LED {
	public:
	typedef enum {
		TYPE_GR = 0,
		TYPE_G_R = 1,
		TYPE_G_O = 2,
	} Type_t;

	typedef enum {
		STATE_OFF = 0,
		STATE_ON,
		STATE_BLINK,
	} State_t;

	typedef enum {
		COLOR_NONE		= 0,

		COLOR_ONE		= 1,
		COLOR_GREEN		= 1,

		COLOR_TWO		= 2,
		COLOR_RED		= 2,
		COLOR_ORANGE		= 2,

		COLOR_BOTH		= 3,
		COLOR_GREEN_ORANGE	= 3,
		COLOR_GREEN_RED		= 3,
		COLOR_YELLOW		= 3
	} Color_t;

	/*
	 * Constructor
	 */
	USL_LED(Type_t type = TYPE_GR, Color_t color = COLOR_NONE,
		State_t state = STATE_OFF);
	USL_LED(USL_LED&& o) noexcept = default;
	~USL_LED();

	/*
	 * Setter / Getter
	 */
	Color_t getcolor(void) const;
	State_t getstate(void) const;

	void set(Color_t color, State_t state = USL_LED::STATE_ON);
	void clear(void);

	const Type_t& gettype(void);
	void settype(Type_t type);

	/*
	 * Public methods
	 */
	void creategpios(const std::vector<std::string> &gpios);

	/*
	 * Time event
	 */
	virtual bool time_event(void);

	/*
	 * Static functions
	 */
	static State_t str_to_state(const std::string &str);
	static const std::string& state_to_str(State_t state);
	static const std::string& state_to_str(const USL_LED &led);

	static Color_t str_to_color(const USL_LED &led, const std::string &str);
	static Color_t str_to_color(const Type_t type, const std::string &str);
	static const std::string& color_to_str(const USL_LED &led);
	static const std::string& color_to_str(const Type_t type, const Color_t color);


	protected:
	Type_t type_;

	/* Color of led */
	Color_t color_;

	/* State of led */
	State_t state_;

	/* True if leds have to be written due to a change */
	bool update_;

	/* Used to save blink state */
	bool blink_on_;

	/* GPIOs used for driving LEDs */
	std::vector<std::unique_ptr<usl_gpio>> gpios_;

	/*
	 * Helpers
	 */
	void led_on(void);
	void led_off(void);

	/*
	 * Static tables for conversion
	 */
	static const std::map<State_t, std::string> STATE_TABLE;

	/* This table must correspond the the Type_t list */
	static const std::map<Color_t, std::string> COLOR_TABLE[];

	static const std::string EMPTY_STR;


	private:
	/* None */
};
